import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { sendTaskReminderEmail } from "@/lib/email";
import { getCurrentPeriodDate } from "@/lib/taskPeriods";

/**
 * API endpoint for sending daily task reminders
 * This should be called by a cron job (e.g., Vercel Cron, GitHub Actions, etc.)
 * at 9:00 AM on weekdays
 */
export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  // Optional: Add authentication/authorization check
  // For example, check for a secret token in headers
  const authToken = req.headers.authorization?.replace("Bearer ", "");
  const expectedToken = process.env.CRON_SECRET || "your-secret-token";

  if (authToken !== expectedToken) {
    return res.status(401).json({ ok: false, error: "Unauthorized" });
  }

  if (req.method !== "POST") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  try {
    const today = new Date();
    const dayOfWeek = today.getDay(); // 0 = Sunday, 6 = Saturday

    // Only send reminders on weekdays (Monday-Friday)
    if (dayOfWeek === 0 || dayOfWeek === 6) {
      return res.status(200).json({
        ok: true,
        message: "Skipping reminder - weekend",
      });
    }

    // Get all technicians
    const technicians = await prisma.user.findMany({
      where: {
        role: {
          equals: "TECHNICIAN",
          mode: "insensitive",
        },
      },
      select: {
        id: true,
        name: true,
        email: true,
      },
    }).then(users => users.filter(user => user.email !== null));

    if (technicians.length === 0) {
      return res.status(200).json({
        ok: true,
        message: "No technicians found",
      });
    }

    const results = [];
    const todayDate = new Date();
    todayDate.setHours(0, 0, 0, 0);

    // Process each technician
    for (const technician of technicians) {
      if (!technician.email) continue;

      // Get incomplete and overdue tasks for this technician
      const tasks = await prisma.task.findMany({
        where: {
          assignedToId: technician.id,
          completedAt: null, // Not completed
        },
        include: {
          itemCompletions: {
            where: {
              periodDate: { not: null },
            },
          },
          periodCompletions: true,
        },
      });

      // Filter tasks to only include incomplete/overdue ones
      const incompleteTasks = [];

      for (const task of tasks) {
        let isIncomplete = false;
        let isOverdue = false;

        if (task.isRecurring && task.recurrencePattern) {
          // For recurring tasks, check if current period is incomplete
          const currentPeriodDate = getCurrentPeriodDate(
            task.recurrencePattern
          );
          const currentPeriodCompletion = task.periodCompletions.find(
            (pc) => pc.periodDate.getTime() === currentPeriodDate.getTime()
          );

          if (!currentPeriodCompletion) {
            isIncomplete = true;
            // Check if period is overdue (past due date)
            if (task.dueDate) {
              const dueDate = new Date(task.dueDate);
              dueDate.setHours(0, 0, 0, 0);
              isOverdue = todayDate > dueDate;
            }
          }
        } else {
          // For non-recurring tasks, check if not completed
          isIncomplete = true;

          // Check if overdue
          if (task.dueDate) {
            const dueDate = new Date(task.dueDate);
            dueDate.setHours(0, 0, 0, 0);
            isOverdue = todayDate > dueDate;
          }
        }

        if (isIncomplete) {
          incompleteTasks.push({
            id: task.id,
            title: task.title,
            description: task.description,
            dueDate: task.dueDate,
            isOverdue,
            isRecurring: task.isRecurring,
            recurrencePattern: task.recurrencePattern,
          });
        }
      }

      // Only send email if there are incomplete/overdue tasks
      if (incompleteTasks.length > 0) {
        try {
          await sendTaskReminderEmail(
            technician.email,
            technician.name || "Technician",
            incompleteTasks
          );
          results.push({
            technician: technician.email,
            tasksCount: incompleteTasks.length,
            status: "sent",
          });
        } catch (error: any) {
          console.error(
            `Failed to send reminder to ${technician.email}:`,
            error
          );
          results.push({
            technician: technician.email,
            tasksCount: incompleteTasks.length,
            status: "failed",
            error: error?.message,
          });
        }
      } else {
        results.push({
          technician: technician.email,
          tasksCount: 0,
          status: "skipped",
        });
      }
    }

    return res.status(200).json({
      ok: true,
      message: "Reminders processed",
      results,
      summary: {
        total: technicians.length,
        sent: results.filter((r) => r.status === "sent").length,
        skipped: results.filter((r) => r.status === "skipped").length,
        failed: results.filter((r) => r.status === "failed").length,
      },
    });
  } catch (error: any) {
    console.error("Error processing task reminders:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to process reminders",
      details: error?.message,
    });
  }
}
